/*
    ElementPostProcess lets you specify post-processing characters for
	OpenDocument elements.
    Copyright (C) 2005  J. David Eisenberg

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
	
	Author: J. David Eisenberg
	Contact: catcode@catcode.com

*/
package com.catcode.odf;

/**
 * Describes what character the <code>OpenDocumentTextInputStream</code>
 * class should emit when an element closes.
 * For example, after closing a <code>&lt;text:p&gt;</code> or
 * <code>&lt;text:h&gt;</code> element, emit a <code>'\n'</code>.
 * If you want to specify an element with no post-processing,
 * set its <code>postProcess</code> field to <code>'\0'</code>
 */
public class ElementPostProcess implements Comparable
{
	protected String name;
	protected char postProcess;
	
	/**
	 * Constructs a placeholder element descriptor.
	 *
	 */
	public ElementPostProcess( )
	{
		this( "", '\0' );
	}
	
	/**
	 * Constructs an element descriptor with the given name and
	 * <code>'\0'</code> post-processing character.
	 *
	 * @param name the element name, without a namespace prefix.
	 */
	public ElementPostProcess( String name )
	{
		this( name, '\0');
	}
	
	/**
	 * Constructs an element descriptor with the given name and
	 * post-processing character.
	 *
	 * @param name the element name, without a namespace prefix.
	 * @param postProcess the character to emit after the element closes.
	 */
	public ElementPostProcess( String name, char postProcess )
	{
		this.name = name;
		this.postProcess = postProcess;
	}
	
	/**
	 * Returns the element's post-processing character.
	 * @return the post-processing character.
	 */
	public char getPostProcess( )
	{
		return postProcess;
	}
	
	/**
	 * Sets element post-processing character to the given value.
	 * @param postProcess the post-processing character for this element.
	 */
	public void setPostProcess( char postProcess )
	{
		this.postProcess = postProcess;
	}

	/**
	 * Returns the element name.
	 * @return The element name.
	 */
	public String getName( )
	{
		return name;
	}
	
	/**
	 * Sets the element name to the given value.
	 * @param name the desired name.
	 */
	public void setName( String name )
	{
		this.name = name;
	}
	
	/**
	 * See <code>Comparator</code> interface for details.
	 */
	public int compareTo(Object o) 
	{
		return this.name.compareTo( ((ElementPostProcess) o).name );
	}
}

